package queue

import (
	"encoding/json"
	"fmt"
	"os"
	"time"

	"github.com/google/uuid"
	"jonasled.dev/jonasled/ems-esp-logger/log"
)

type QueueElement struct {
	ID          string    `json:"id"`           // Unique identifier for the job
	Value       string    `json:"value"`        // Job data
	EnqueueTime time.Time `json:"enqueue_time"` // Time the job was enqueued
	MinDuration int       `json:"min_duration"` // Minimum duration in seconds
	Done        bool      `json:"done"`         // Job completion status
}

type Queue struct {
	Elements []QueueElement
	FilePath string
}

func NewQueue(filePath string) (*Queue, error) {
	q := &Queue{
		Elements: make([]QueueElement, 0),
		FilePath: filePath,
	}

	// Load state from file
	if _, err := os.Stat(filePath); err == nil {
		data, err := os.ReadFile(filePath)
		if err != nil {
			return nil, fmt.Errorf("error reading state file: %w", err)
		}
		if err := json.Unmarshal(data, &q.Elements); err != nil {
			return nil, fmt.Errorf("error parsing state file: %w", err)
		}

		// Set all jobs' Done status to false when loading the queue
		for i := range q.Elements {
			q.Elements[i].Done = false
		}

		log.Log.Info("Initialized queue, current size: ", q.GetCurrentSize())
	}
	return q, nil
}

func (q *Queue) Enqueue(element string, minDuration int) {
	// Generate a unique ID for the new job
	id := uuid.New().String()

	q.Elements = append(q.Elements, QueueElement{
		ID:          id,
		Value:       element,
		EnqueueTime: time.Now(),
		MinDuration: minDuration,
		Done:        false, // Job is not done initially
	})
}

func (q *Queue) Dequeue() (string, string, error) {
	if len(q.Elements) == 0 {
		return "", "", fmt.Errorf("queue is empty")
	}

	currentTime := time.Now()
	for i, elem := range q.Elements {
		// Check if the element has satisfied its minimum duration and is not marked as done
		if currentTime.Sub(elem.EnqueueTime).Seconds() >= float64(elem.MinDuration) && !elem.Done {
			// Lock the job for this task runner by marking it as "done"
			q.Elements[i].Done = true
			// Return job value and ID to the task runner
			return elem.Value, elem.ID, nil
		}
	}

	return "", "", fmt.Errorf("no elements are ready for dequeuing")
}

func (q *Queue) MarkDone(id string) error {
	// Find the job in the queue by ID and mark it as done
	for i, elem := range q.Elements {
		if elem.ID == id && elem.Done {
			// Once marked as done, remove the job from the queue
			q.Elements = append(q.Elements[:i], q.Elements[i+1:]...)
			return nil
		}
	}
	return fmt.Errorf("job with ID %s not found or already marked as done", id)
}

func (q *Queue) GetCurrentSize() int {
	return len(q.Elements)
}

func (q *Queue) Save() error {
	data, err := json.Marshal(q.Elements)
	if err != nil {
		return fmt.Errorf("error serializing state: %w", err)
	}
	if err := os.WriteFile(q.FilePath, data, 0644); err != nil {
		return fmt.Errorf("error writing state to file: %w", err)
	}
	return nil
}
