package database

import (
	"fmt"
	"os"
	"time"

	"github.com/glebarez/sqlite"
	gormloggerlogrus "github.com/nekomeowww/gorm-logger-logrus"
	"github.com/sirupsen/logrus"
	"gorm.io/driver/mysql"
	"gorm.io/driver/postgres"
	"gorm.io/gorm"
	"gorm.io/gorm/logger"
	"jonasled.dev/jonasled/ems-esp-logger/database/tables"
	"jonasled.dev/jonasled/ems-esp-logger/log"
)

var Db *gorm.DB

func Init() {
	var err error
	var dbdriver gorm.Dialector
	switch os.Getenv("OUTPUT_DATABASE_TYPE") {
	case "sqlite":
		dbdriver = sqlite.Open(os.Getenv("OUTPUT_DATABSE"))
	case "mysql":
		dsn := fmt.Sprintf("%s:%s@tcp(%s)/%s?charset=utf8mb4&parseTime=True&loc=Local",
			os.Getenv("OUTPUT_DATABASE_USER"),
			os.Getenv("OUTPUT_DATABASE_PASSWORD"),
			os.Getenv("OUTPUT_DATABASE_HOST"),
			os.Getenv("OUTPUT_DATABSE"),
		)
		dbdriver = mysql.Open(dsn)
	case "postgres":
		dsn := fmt.Sprintf("host=%s user=%s password=%s dbname=%s port=5432 sslmode=disable TimeZone=UTC",
			os.Getenv("OUTPUT_DATABASE_HOST"),
			os.Getenv("OUTPUT_DATABASE_USER"),
			os.Getenv("OUTPUT_DATABASE_PASSWORD"),
			os.Getenv("OUTPUT_DATABSE"),
		)
		dbdriver = postgres.Open(dsn)
	default:
		log.Log.Fatal("Please set OUTPUT_DATABASE_TYPE to a valid value")
	}
	Db, err = gorm.Open(dbdriver, &gorm.Config{
		Logger: gormloggerlogrus.New(gormloggerlogrus.Options{
			Logger:                    logrus.NewEntry(log.Log),
			LogLevel:                  logger.Error,
			IgnoreRecordNotFoundError: false,
			SlowThreshold:             time.Millisecond * 200,
			FileWithLineNumField:      "file",
		}),
	})

	if err != nil {
		log.Log.Fatal("Failed initializing database: ", err.Error())
	}

	if os.Getenv("OUTPUT_DATABASE_EXECUTE_MIGRATIONS") != "false" {
		log.Log.Info("Executing database migrations")
		Db.AutoMigrate(&tables.Instance{}, &tables.ValueType{}, &tables.Value{})
	}
}
